"""
Singleton Pattern for AI-AntiBot Components
Ensures single instance of training monitor to maintain data persistence
"""
import threading
from typing import Optional

class SingletonMeta(type):
    """Metaclass that creates a Singleton base class when called."""
    _instances = {}
    _lock: threading.Lock = threading.Lock()

    def __call__(cls, *args, **kwargs):
        """
        Possible changes to the value of the `__init__` argument do not affect
        the returned instance.
        """
        if cls not in cls._instances:
            with cls._lock:
                if cls not in cls._instances:
                    instance = super().__call__(*args, **kwargs)
                    cls._instances[cls] = instance
        return cls._instances[cls]

class TrainingMonitorSingleton:
    """Singleton wrapper for TrainingMonitor to ensure data persistence"""
    _instance: Optional['TrainingMonitor'] = None
    _lock = threading.Lock()
    
    @classmethod
    def get_instance(cls):
        """Get the singleton instance of TrainingMonitor"""
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    from .dashboard.monitor import TrainingMonitor
                    cls._instance = TrainingMonitor()
        return cls._instance
    
    @classmethod
    def reset_instance(cls):
        """Reset the singleton instance (for testing)"""
        with cls._lock:
            cls._instance = None