"""
AI-AntiBot Dashboard Routes
Flask routes for the AI-AntiBot training dashboard
"""
from flask import Blueprint, render_template, jsonify, request
from flask_login import login_required
import logging

from ai_antibot.singleton import TrainingMonitorSingleton
from ai_antibot.training.trainer import TurnstileTrainer
from ai_antibot.models.detector import BotDetector

logger = logging.getLogger(__name__)

# Create blueprint
ai_dashboard_bp = Blueprint('ai_dashboard', __name__, url_prefix='/ai-dashboard')

# Initialize components
training_monitor = TrainingMonitorSingleton.get_instance()
trainer = TurnstileTrainer()
detector = BotDetector()

@ai_dashboard_bp.route('/')
@login_required
def dashboard():
    """Main AI-AntiBot dashboard page"""
    return render_template('ai_dashboard/monitor.html')

@ai_dashboard_bp.route('/data')
@login_required
def dashboard_data():
    """Get comprehensive dashboard data"""
    try:
        data = training_monitor.get_dashboard_data()
        return jsonify(data)
    except Exception as e:
        logger.error(f"Error getting dashboard data: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/feed')
@login_required
def live_feed():
    """Get live feed events"""
    try:
        since_id = request.args.get('since', 0, type=int)
        events = training_monitor.get_live_feed(since_id)
        return jsonify(events)
    except Exception as e:
        logger.error(f"Error getting live feed: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/force-training', methods=['POST'])
@login_required
def force_training():
    """Force immediate model training"""
    try:
        result = trainer.force_training()
        if result['success']:
            # Record training event in monitor
            training_monitor.record_model_training(result)
        return jsonify(result)
    except Exception as e:
        logger.error(f"Error forcing training: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@ai_dashboard_bp.route('/stats')
@login_required
def training_stats():
    """Get detailed training statistics"""
    try:
        stats = trainer.get_training_stats()
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Error getting training stats: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/model-stats')
@login_required
def model_stats():
    """Get model performance statistics"""
    try:
        stats = detector.get_model_stats()
        return jsonify(stats)
    except Exception as e:
        logger.error(f"Error getting model stats: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/feature-importance')
@login_required
def feature_importance():
    """Get feature importance data"""
    try:
        importance = detector.get_feature_importance()
        return jsonify(importance)
    except Exception as e:
        logger.error(f"Error getting feature importance: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/visualization-data')
@login_required
def visualization_data():
    """Get data for training visualizations"""
    try:
        data = training_monitor.get_training_visualization_data()
        return jsonify(data)
    except Exception as e:
        logger.error(f"Error getting visualization data: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/export-training-data')
@login_required
def export_training_data():
    """Export training data"""
    try:
        format_type = request.args.get('format', 'json')
        data = trainer.export_training_data(format_type)
        
        if format_type == 'json':
            return jsonify({'data': data})
        elif format_type == 'csv':
            from flask import Response
            return Response(
                data,
                mimetype='text/csv',
                headers={'Content-Disposition': 'attachment; filename=training_data.csv'}
            )
        else:
            return jsonify({'error': 'Unsupported format'}), 400
            
    except Exception as e:
        logger.error(f"Error exporting training data: {e}")
        return jsonify({'error': str(e)}), 500

@ai_dashboard_bp.route('/clear-training-data', methods=['POST'])
@login_required
def clear_training_data():
    """Clear all training data (for testing)"""
    try:
        trainer.clear_training_data()
        return jsonify({'success': True, 'message': 'Training data cleared'})
    except Exception as e:
        logger.error(f"Error clearing training data: {e}")
        return jsonify({'success': False, 'error': str(e)}), 500

@ai_dashboard_bp.route('/system-health')
@login_required
def system_health():
    """Get system health status"""
    try:
        health = training_monitor._get_system_status()
        return jsonify(health)
    except Exception as e:
        logger.error(f"Error getting system health: {e}")
        return jsonify({'error': str(e)}), 500