import os
import logging
import json
from flask import Flask
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager
from sqlalchemy.orm import DeclarativeBase
from werkzeug.middleware.proxy_fix import ProxyFix

# Configure logging
logging.basicConfig(level=logging.DEBUG)

class Base(DeclarativeBase):
    pass

# Initialize extensions
db = SQLAlchemy(model_class=Base)
login_manager = LoginManager()

def create_app():
    app = Flask(__name__)
    
    # Configuration
    app.secret_key = os.environ.get("SESSION_SECRET") or "dev-secret-key-change-in-production"
    if not app.secret_key or app.secret_key == "dev-secret-key-change-in-production":
        logging.warning("Using default secret key. Set SESSION_SECRET environment variable for production.")
    app.wsgi_app = ProxyFix(app.wsgi_app, x_proto=1, x_host=1)
    
    # Database configuration - SQLite3
    database_path = os.path.join(os.path.dirname(__file__), 'instance', 'database.db')
    os.makedirs(os.path.dirname(database_path), exist_ok=True)
    app.config["SQLALCHEMY_DATABASE_URI"] = f"sqlite:///{database_path}"
    app.config["SQLALCHEMY_TRACK_MODIFICATIONS"] = False
    
    # Initialize extensions
    db.init_app(app)
    login_manager.init_app(app)
    login_manager.login_view = 'auth.login'
    
    # Add custom template filters for JSON parsing
    @app.template_filter('parse_json')
    def parse_json_filter(value):
        """Parse JSON string to Python object"""
        if value:
            try:
                return json.loads(value)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    # Register blueprints
    from routes import main_bp
    from auth import auth_bp
    from api import api_bp
    from ai_dashboard_routes import ai_dashboard_bp
    
    app.register_blueprint(main_bp)
    app.register_blueprint(auth_bp, url_prefix='/auth')
    app.register_blueprint(api_bp, url_prefix='/api')
    app.register_blueprint(ai_dashboard_bp)
    
    # Create tables
    with app.app_context():
        import models
        db.create_all()
        
        # Create default admin user if not exists
        from models import AdminUser
        from werkzeug.security import generate_password_hash
        
        admin = AdminUser.query.filter_by(username='admin').first()
        if not admin:
            admin = AdminUser(
                username='admin',
                password_hash=generate_password_hash('admin')
            )
            db.session.add(admin)
            db.session.commit()
            logging.info("Default admin user created")
        
        # Discover and sync app templates
        from utils import discover_and_sync_templates
        discover_and_sync_templates()
        

    
    return app

# Create app instance
app = create_app()
