#!/usr/bin/env python3
"""
SQLite3 database migration script to add multi-step fields
"""

import sqlite3
import os
import sys

def migrate_database():
    """Add multi-step fields to SQLite3 database"""
    
    # Database path
    db_path = 'instance/database.db'
    
    if not os.path.exists(db_path):
        print("Database file not found. Please run the application first to create the database.")
        return False
    
    try:
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        print("Starting SQLite3 database migration...")
        
        # Check if columns already exist in deployed_apps table
        cursor.execute("PRAGMA table_info(deployed_apps)")
        columns = [column[1] for column in cursor.fetchall()]
        
        # Add multi_step_enabled column if it doesn't exist
        if 'multi_step_enabled' not in columns:
            cursor.execute('ALTER TABLE deployed_apps ADD COLUMN multi_step_enabled INTEGER DEFAULT 0')
            print("Added multi_step_enabled column to deployed_apps table")
        
        # Add step_count column if it doesn't exist
        if 'step_count' not in columns:
            cursor.execute('ALTER TABLE deployed_apps ADD COLUMN step_count INTEGER DEFAULT 1')
            print("Added step_count column to deployed_apps table")
        
        # Add flow_config column if it doesn't exist
        if 'flow_config' not in columns:
            cursor.execute('ALTER TABLE deployed_apps ADD COLUMN flow_config TEXT')
            print("Added flow_config column to deployed_apps table")
        
        # Check if columns already exist in form_submissions table
        cursor.execute("PRAGMA table_info(form_submissions)")
        columns = [column[1] for column in cursor.fetchall()]
        
        # Add session_id column if it doesn't exist
        if 'session_id' not in columns:
            cursor.execute('ALTER TABLE form_submissions ADD COLUMN session_id TEXT')
            print("Added session_id column to form_submissions table")
        
        # Add step_count column if it doesn't exist
        if 'step_count' not in columns:
            cursor.execute('ALTER TABLE form_submissions ADD COLUMN step_count INTEGER DEFAULT 1')
            print("Added step_count column to form_submissions table")
        
        # Add completion_time column if it doesn't exist
        if 'completion_time' not in columns:
            cursor.execute('ALTER TABLE form_submissions ADD COLUMN completion_time TEXT')
            print("Added completion_time column to form_submissions table")
        
        # Commit changes
        conn.commit()
        print("SQLite3 database migration completed successfully!")
        return True
        
    except sqlite3.Error as e:
        print(f"Database error during migration: {e}")
        return False
    except Exception as e:
        print(f"Unexpected error during migration: {e}")
        return False
    finally:
        if conn:
            conn.close()

if __name__ == "__main__":
    if migrate_database():
        print("Migration completed successfully!")
        sys.exit(0)
    else:
        print("Migration failed!")
        sys.exit(1)