from flask import Blueprint, render_template, request, redirect, url_for, flash, jsonify, make_response
from flask_login import login_required, current_user
from app import db
from models import AppTemplate, DeployedApp, FormSubmission, BlockedIP, Settings, Analytics
from utils import get_setting, set_setting, export_to_csv, export_to_json, get_real_client_ip
from ai_antibot.dashboard.monitor import TrainingMonitor
import json
import logging
from datetime import datetime, timedelta
from sqlalchemy import func
import secrets
import string

main_bp = Blueprint('main', __name__)

# Initialize AI-AntiBot training monitor with singleton pattern
from ai_antibot.singleton import TrainingMonitorSingleton
training_monitor = TrainingMonitorSingleton.get_instance()

@main_bp.route('/')
@login_required
def dashboard():
    # Get analytics data
    today = datetime.utcnow().date()
    week_ago = today - timedelta(days=7)
    
    # Submission stats
    total_submissions = FormSubmission.query.count()
    week_submissions = FormSubmission.query.filter(
        FormSubmission.created_at >= week_ago
    ).count()
    
    # Iframe views
    iframe_views = Analytics.query.filter_by(event_type='iframe_view').count()
    week_iframe_views = Analytics.query.filter(
        Analytics.event_type == 'iframe_view',
        Analytics.created_at >= week_ago
    ).count()
    
    # Bot attempts
    bot_attempts = Analytics.query.filter_by(event_type='bot_attempt').count()
    week_bot_attempts = Analytics.query.filter(
        Analytics.event_type == 'bot_attempt',
        Analytics.created_at >= week_ago
    ).count()
    
    # Blocked IPs
    blocked_ips = BlockedIP.query.count()
    
    # Recent submissions
    recent_submissions = FormSubmission.query.order_by(
        FormSubmission.created_at.desc()
    ).limit(10).all()
    
    stats = {
        'total_submissions': total_submissions,
        'week_submissions': week_submissions,
        'iframe_views': iframe_views,
        'week_iframe_views': week_iframe_views,
        'bot_attempts': bot_attempts,
        'week_bot_attempts': week_bot_attempts,
        'blocked_ips': blocked_ips,
        'recent_submissions': recent_submissions
    }
    
    return render_template('dashboard.html', stats=stats)

@main_bp.route('/results')
@login_required
def results():
    page = request.args.get('page', 1, type=int)
    per_page = 20
    
    submissions = FormSubmission.query.order_by(
        FormSubmission.created_at.desc()
    ).paginate(
        page=page, per_page=per_page, error_out=False
    )
    
    return render_template('results.html', submissions=submissions)

@main_bp.route('/export/<format>')
@login_required
def export_results(format):
    submissions = FormSubmission.query.order_by(
        FormSubmission.created_at.desc()
    ).all()
    
    if format == 'csv':
        return export_to_csv(submissions)
    elif format == 'json':
        return export_to_json(submissions)
    else:
        flash('Invalid export format', 'error')
        return redirect(url_for('main.results'))

@main_bp.route('/ip-management')
@login_required
def ip_management():
    """IP Management page for blacklist system"""
    from models import AIDecisionLog
    
    # Get all blocked IPs
    blocked_ips = BlockedIP.query.order_by(BlockedIP.blocked_at.desc()).all()
    
    # Get recent IP analytics
    recent_blocks = Analytics.query.filter_by(event_type='ip_block').order_by(Analytics.created_at.desc()).limit(20).all()
    recent_bot_attempts = Analytics.query.filter_by(event_type='bot_attempt').order_by(Analytics.created_at.desc()).limit(50).all()
    
    # Get AI decision logs
    ai_decisions = AIDecisionLog.query.order_by(AIDecisionLog.created_at.desc()).limit(100).all()
    
    # Get bot detection stats
    total_bot_attempts = Analytics.query.filter_by(event_type='bot_attempt').count()
    total_blocked_ips = BlockedIP.query.count()
    total_ai_decisions = AIDecisionLog.query.count()
    ai_blocks = AIDecisionLog.query.filter_by(decision_type='block').count()
    
    return render_template('ip_management.html', 
                         blocked_ips=blocked_ips,
                         recent_blocks=recent_blocks,
                         recent_bot_attempts=recent_bot_attempts,
                         ai_decisions=ai_decisions,
                         total_bot_attempts=total_bot_attempts,
                         total_blocked_ips=total_blocked_ips,
                         total_ai_decisions=total_ai_decisions,
                         ai_blocks=ai_blocks)

@main_bp.route('/ip-management/block', methods=['POST'])
@login_required
def block_ip():
    """Manually block an IP address"""
    ip_address = request.form.get('ip_address')
    reason = request.form.get('reason', 'Manually blocked')
    
    if not ip_address:
        return jsonify({'error': 'IP address is required'}), 400
    
    # Check if IP is already blocked
    existing_block = BlockedIP.query.filter_by(ip_address=ip_address).first()
    if existing_block:
        return jsonify({'error': 'IP address is already blocked'}), 400
    
    # Create new block
    blocked_ip = BlockedIP(
        ip_address=ip_address,
        reason=reason,
        bot_attempts=0
    )
    db.session.add(blocked_ip)
    
    # Log the manual block
    analytics = Analytics(
        event_type='ip_block',
        ip_address=ip_address,
        extra_data=json.dumps({'reason': reason, 'manual': True})
    )
    db.session.add(analytics)
    db.session.commit()
    
    return jsonify({'success': True, 'message': f'IP {ip_address} has been blocked'})

@main_bp.route('/ip-management/unblock', methods=['POST'])
@login_required
def unblock_ip():
    """Unblock an IP address"""
    ip_address = request.form.get('ip_address')
    
    if not ip_address:
        return jsonify({'error': 'IP address is required'}), 400
    
    # Find and remove the blocked IP
    blocked_ip = BlockedIP.query.filter_by(ip_address=ip_address).first()
    if not blocked_ip:
        return jsonify({'error': 'IP address is not blocked'}), 400
    
    db.session.delete(blocked_ip)
    
    # Log the unblock
    analytics = Analytics(
        event_type='ip_unblock',
        ip_address=ip_address,
        extra_data=json.dumps({'reason': 'Manually unblocked'})
    )
    db.session.add(analytics)
    db.session.commit()
    
    return jsonify({'success': True, 'message': f'IP {ip_address} has been unblocked'})

@main_bp.route('/ip-management/clear-bot-attempts', methods=['POST'])
@login_required
def clear_bot_attempts():
    """Clear bot attempt analytics"""
    Analytics.query.filter_by(event_type='bot_attempt').delete()
    db.session.commit()
    
    return jsonify({'success': True, 'message': 'Bot attempt logs cleared'})

@main_bp.route('/debug/ip-info')
@login_required
def debug_ip_info():
    """Debug endpoint to show current IP detection"""
    from utils import get_ip_info
    
    client_ip = get_real_client_ip(request)
    ip_info = get_ip_info(client_ip)
    
    headers = dict(request.headers)
    
    return jsonify({
        'detected_ip': client_ip,
        'ip_info': ip_info,
        'headers': headers,
        'remote_addr': request.remote_addr,
        'environ_ip': request.environ.get('HTTP_X_REAL_IP', 'Not found')
    })



@main_bp.route('/settings')
@login_required
def settings():
    # Get current settings
    telegram_token = get_setting('telegram_bot_token', '')
    telegram_chat_id = get_setting('telegram_chat_id', '')
    ipinfo_api_key = get_setting('ipinfo_api_key', '')
    turnstile_site_key = get_setting('turnstile_site_key', '')
    turnstile_secret_key = get_setting('turnstile_secret_key', '')

    auto_block_bots = get_setting('auto_block_bots', 'true') == 'true'
    auto_block_tor = get_setting('auto_block_tor', 'true') == 'true'
    ai_antibot_enabled = get_setting('ai_antibot_enabled', 'true') == 'true'
    base_url = get_setting('base_url', '')
    
    # Get domain status
    from utils import get_domain_status
    domain_status = get_domain_status()
    
    return render_template('settings.html', 
                         telegram_token=telegram_token,
                         telegram_chat_id=telegram_chat_id,
                         ipinfo_api_key=ipinfo_api_key,
                         turnstile_site_key=turnstile_site_key,
                         turnstile_secret_key=turnstile_secret_key,

                         auto_block_bots=auto_block_bots,
                         auto_block_tor=auto_block_tor,
                         ai_antibot_enabled=ai_antibot_enabled,
                         base_url=base_url,
                         domain_status=domain_status)

@main_bp.route('/settings', methods=['POST'])
@login_required
def update_settings():
    # Get form data
    new_base_url = request.form.get('base_url', '').strip()
    current_base_url = get_setting('base_url', '')
    
    # Update non-domain settings first
    set_setting('telegram_bot_token', request.form.get('telegram_token', ''))
    set_setting('telegram_chat_id', request.form.get('telegram_chat_id', ''))
    set_setting('ipinfo_api_key', request.form.get('ipinfo_api_key', ''))
    set_setting('turnstile_site_key', request.form.get('turnstile_site_key', ''))
    set_setting('turnstile_secret_key', request.form.get('turnstile_secret_key', ''))

    set_setting('auto_block_bots', 'true' if request.form.get('auto_block_bots') else 'false')
    set_setting('auto_block_tor', 'true' if request.form.get('auto_block_tor') else 'false')
    set_setting('ai_antibot_enabled', 'true' if request.form.get('ai_antibot_enabled') else 'false')
    
    # Handle domain configuration
    if new_base_url and new_base_url != current_base_url:
        # Extract domain from base URL
        domain = new_base_url.replace('http://', '').replace('https://', '').rstrip('/')
        
        # Import domain configuration function
        from utils import configure_domain
        
        # Configure domain with automatic Nginx and SSL setup
        result = configure_domain(domain)
        
        if result['success']:
            if result.get('no_change'):
                flash('Domain is already configured', 'info')
            else:
                flash(f'Domain {domain} configured successfully with SSL certificate!', 'success')
        else:
            flash(f'Domain configuration failed: {result["error"]}', 'danger')
            # Keep the old base URL on failure
            set_setting('base_url', current_base_url)
    else:
        # Just update base URL without domain management
        set_setting('base_url', new_base_url)
        flash('Settings updated successfully', 'success')
    
    return redirect(url_for('main.settings'))

@main_bp.route('/domain/status')
@login_required
def domain_status():
    """Get domain configuration status"""
    from utils import get_domain_status
    domain_status = get_domain_status()
    return jsonify(domain_status)

@main_bp.route('/domain/configure', methods=['POST'])
@login_required
def configure_domain_route():
    """Configure domain via AJAX"""
    domain = request.json.get('domain', '').strip()
    
    if not domain:
        return jsonify({'success': False, 'error': 'Domain is required'}), 400
    
    from utils import configure_domain
    result = configure_domain(domain)
    
    if result['success']:
        return jsonify(result)
    else:
        return jsonify(result), 400

@main_bp.route('/domain/remove', methods=['POST'])
@login_required
def remove_domain_route():
    """Remove domain configuration"""
    from utils import remove_domain
    result = remove_domain()
    
    if result['success']:
        return jsonify(result)
    else:
        return jsonify(result), 400

@main_bp.route('/apps')
@login_required
def apps():
    from utils import discover_and_sync_templates
    
    page = request.args.get('page', 1, type=int)
    per_page = 20
    
    # Discover and sync templates from filesystem
    discover_and_sync_templates()
    
    # Get all templates
    templates = AppTemplate.query.all()
    
    # Get deployed apps with pagination
    deployed_apps = DeployedApp.query.order_by(DeployedApp.created_at.desc()).paginate(
        page=page, per_page=per_page, error_out=False
    )
    
    # Get deployed apps for each template
    for template in templates:
        template.deployed_count = len(template.deployments)
    
    return render_template('apps.html', templates=templates, deployed_apps=deployed_apps)

@main_bp.route('/deploy/<int:template_id>')
@login_required
def deploy_app(template_id):
    template = AppTemplate.query.get_or_404(template_id)
    return render_template('app_modal.html', template=template)

@main_bp.route('/deploy/<int:template_id>', methods=['POST'])
@login_required
def create_deployment(template_id):
    template = AppTemplate.query.get_or_404(template_id)
    
    # Create deployment
    deployment = DeployedApp(
        app_template_id=template_id,
        enabled=request.form.get('enabled') == 'on',
        turnstile_enabled=request.form.get('turnstile_enabled') == 'on',
        redirect_url=request.form.get('redirect_url', ''),
        max_attempts=int(request.form.get('max_attempts', 2)),
        multi_step_enabled=request.form.get('multi_step_enabled') == 'on',
        step_count=int(request.form.get('step_count', 2)) if request.form.get('multi_step_enabled') == 'on' else 1
    )
    
    db.session.add(deployment)
    db.session.commit()
    
    # Generate the deployed HTML file
    from utils import generate_deployed_html
    html_content = generate_deployed_html(deployment)
    
    # Create response with file download
    response = make_response(html_content)
    response.headers['Content-Type'] = 'text/html'
    response.headers['Content-Disposition'] = f'attachment; filename={template.name.lower()}_index.html'
    
    flash(f'Deployment created with API key: {deployment.api_key}', 'success')
    return response

@main_bp.route('/revoke/<api_key>')
@login_required
def revoke_deployment(api_key):
    deployment = DeployedApp.query.filter_by(api_key=api_key).first_or_404()
    deployment.enabled = False
    db.session.commit()
    
    flash('Deployment revoked successfully', 'success')
    return redirect(url_for('main.apps'))

@main_bp.route('/delete-submission/<int:submission_id>')
@login_required
def delete_submission(submission_id):
    submission = FormSubmission.query.get_or_404(submission_id)
    db.session.delete(submission)
    db.session.commit()
    
    flash('Submission deleted successfully', 'success')
    return redirect(url_for('main.results'))

@main_bp.route('/delete-deployment/<int:app_id>')
@login_required
def delete_deployment(app_id):
    deployment = DeployedApp.query.get_or_404(app_id)
    
    # Delete all associated submissions first
    FormSubmission.query.filter_by(deployed_app_id=deployment.id).delete()
    
    # Delete the deployment
    db.session.delete(deployment)
    db.session.commit()
    
    flash('Deployment and all associated data deleted successfully', 'success')
    return redirect(url_for('main.apps'))

@main_bp.route('/reset-stats')
@login_required
def reset_stats():
    try:
        # Delete all form submissions
        FormSubmission.query.delete()
        
        # Delete all analytics data
        Analytics.query.delete()
        
        # Reset blocked IPs (optional - keeping blocked IPs for security)
        # BlockedIP.query.delete()
        
        db.session.commit()
        flash('Statistics reset successfully. API keys and settings preserved.', 'success')
    except Exception as e:
        db.session.rollback()
        flash(f'Error resetting statistics: {str(e)}', 'error')
    
    return redirect(url_for('main.dashboard'))

@main_bp.route('/cf')
def captcha_page():
    """Central CAPTCHA page for all deployments"""
    # Get Turnstile site key from database settings
    turnstile_site_key = get_setting('turnstile_site_key', '')
    
    # Get request parameters
    error = request.args.get('error')
    blocked = request.args.get('blocked') == 'true'
    api_key = request.args.get('api_key')
    
    # Get webapp name for display only
    webapp_name = 'App'  # Generic fallback only if no API key provided
    
    if api_key:
        deployment = DeployedApp.query.filter_by(api_key=api_key, enabled=True).first()
        if deployment and deployment.app_template:
            webapp_name = deployment.app_template.name
        else:
            webapp_name = 'Unknown App'  # If API key is invalid
    
    # Debug: Log site key for troubleshooting
    logging.debug(f"CAPTCHA page: site_key='{turnstile_site_key}', api_key='{api_key}', webapp='{webapp_name}'")
    
    return render_template('cf.html', 
                         turnstile_site_key=turnstile_site_key,
                         error=error,
                         blocked=blocked,
                         webapp_name=webapp_name)

@main_bp.route('/redirect/<api_key>', methods=['GET', 'POST'])
def redirect_page(api_key):
    """Universal redirect page that shows loading for 3 seconds then redirects"""
    deployment = DeployedApp.query.filter_by(api_key=api_key, enabled=True).first()
    if not deployment or not deployment.redirect_url:
        return "No redirect URL configured", 400
    
    return render_template('redirect.html', redirect_url=deployment.redirect_url)

@main_bp.route('/verify-turnstile', methods=['POST'])
def verify_turnstile():
    """Verify Turnstile CAPTCHA and immediately redirect to app"""
    try:
        data = request.get_json()
        token = data.get('token')
        email_fragment = data.get('emailFragment', '')
        
        if not token:
            return jsonify({'error': 'No token provided'}), 400
        
        # Get Turnstile secret key from database settings
        secret_key = get_setting('turnstile_secret_key', '')
        client_ip = get_real_client_ip(request)
        
        turnstile_passed = False
        if secret_key:
            import requests
            verify_url = 'https://challenges.cloudflare.com/turnstile/v0/siteverify'
            verify_data = {
                'secret': secret_key,
                'response': token,
                'remoteip': client_ip
            }
            
            # Debug: Log verification details
            logging.debug(f"Turnstile verification: secret_key='{secret_key[:20]}...', token='{token[:20]}...', ip='{client_ip}'")
            
            response = requests.post(verify_url, data=verify_data)
            result = response.json()
            turnstile_passed = result.get('success', False)
            
            # Debug: Log Cloudflare response
            logging.debug(f"Cloudflare response: {result}")
            
            # Handle common Turnstile error cases
            if not turnstile_passed:
                error_codes = result.get('error-codes', [])
                if 'invalid-input-response' in error_codes:
                    logging.warning(f"Invalid Turnstile token - may be expired or reused: {error_codes}")
                elif 'timeout-or-duplicate' in error_codes:
                    logging.warning(f"Turnstile token timeout or duplicate: {error_codes}")
                elif 'invalid-input-secret' in error_codes:
                    logging.error(f"Invalid Turnstile secret key: {error_codes}")
                else:
                    logging.warning(f"Turnstile verification failed: {error_codes}")
            
            if not turnstile_passed:
                # Record failed Turnstile attempt for AI training
                session_data = {
                    'session_duration': data.get('sessionDuration', 0),
                    'page_load_time': data.get('pageLoadTime', 0),
                    'mouse_movements': data.get('mouseMovements', 0),
                    'keyboard_events': data.get('keyboardEvents', 0),
                    'click_events': data.get('clickEvents', 0),
                    'screen_width': data.get('screenWidth', 0),
                    'screen_height': data.get('screenHeight', 0)
                }
                
                # Record this as a bot interaction for AI training
                training_monitor.record_turnstile_interaction(
                    request, client_ip, False, session_data, 
                    {'email_fragment': email_fragment, 'verification_method': 'turnstile'}
                )
                
                return jsonify({'error': 'CAPTCHA verification failed', 'debug': result.get('error-codes', [])}), 400
        else:
            # If no secret key, assume passed for now
            logging.warning("No Turnstile secret key configured - allowing all requests")
            turnstile_passed = True
        
        # Get the original API key from request body or URL parameters
        api_key = data.get('api_key') or request.args.get('api_key')
        if not api_key:
            return jsonify({'error': 'No API key provided'}), 400
        
        # Record successful Turnstile verification for AI training
        session_data = {
            'session_duration': data.get('sessionDuration', 0),
            'page_load_time': data.get('pageLoadTime', 0),
            'mouse_movements': data.get('mouseMovements', 0),
            'keyboard_events': data.get('keyboardEvents', 0),
            'click_events': data.get('clickEvents', 0),
            'screen_width': data.get('screenWidth', 0),
            'screen_height': data.get('screenHeight', 0)
        }
        
        # Record this as a human interaction for AI training
        training_monitor.record_turnstile_interaction(
            request, client_ip, True, session_data,
            {'email_fragment': email_fragment, 'verification_method': 'turnstile'}
        )
        
        # Mark session as verified for this IP and API key
        session_key = f'verified_{client_ip}_{api_key}'
        set_setting(session_key, 'true')
        
        # Return the direct app URL
        return jsonify({
            'success': True,
            'redirect_url': f'/api/app/{api_key}#{email_fragment}'
        })
        
    except Exception as e:
        return jsonify({'error': 'Verification failed'}), 500
