// Admin Panel JavaScript - Light Grey & Black Theme
// Handles sidebar navigation, modals, forms, and interactive elements

document.addEventListener('DOMContentLoaded', function() {
    
    // Sidebar Toggle Functionality
    const menuToggle = document.getElementById('menu-toggle');
    const sidebar = document.getElementById('sidebar-wrapper');
    const pageContent = document.getElementById('page-content-wrapper');
    
    if (menuToggle) {
        menuToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            pageContent.classList.toggle('sidebar-collapsed');
        });
    }
    
    // Auto-dismiss alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(function(alert) {
        setTimeout(function() {
            if (alert.classList.contains('alert-dismissible')) {
                const closeBtn = alert.querySelector('.btn-close');
                if (closeBtn) {
                    closeBtn.click();
                }
            }
        }, 5000);
    });
    
    // Form validation and enhancement
    const forms = document.querySelectorAll('form');
    forms.forEach(function(form) {
        form.addEventListener('submit', function(e) {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                
                // Re-enable button after 3 seconds (in case of error)
                setTimeout(function() {
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = submitBtn.getAttribute('data-original-text') || 'Submit';
                }, 3000);
            }
        });
        
        // Store original button text
        const submitBtn = form.querySelector('button[type="submit"]');
        if (submitBtn) {
            submitBtn.setAttribute('data-original-text', submitBtn.innerHTML);
        }
    });
    
    // Settings tabs functionality
    const settingsTabs = document.querySelectorAll('#settingsTabs button[data-bs-toggle="tab"]');
    settingsTabs.forEach(function(tab) {
        tab.addEventListener('click', function() {
            // Store active tab in localStorage
            localStorage.setItem('activeSettingsTab', this.id);
        });
    });
    
    // Restore active settings tab
    const activeTab = localStorage.getItem('activeSettingsTab');
    if (activeTab) {
        const tabElement = document.getElementById(activeTab);
        if (tabElement) {
            const tab = new bootstrap.Tab(tabElement);
            tab.show();
        }
    }
    
    // App deployment modal enhancements
    const appCards = document.querySelectorAll('.app-card');
    appCards.forEach(function(card) {
        card.addEventListener('click', function() {
            const cardBody = this.querySelector('.card-body');
            const appName = cardBody.querySelector('.card-title').textContent;
            
            // Add loading state
            card.classList.add('loading');
            setTimeout(function() {
                card.classList.remove('loading');
            }, 500);
        });
    });
    
    // Modal form validation
    const modals = document.querySelectorAll('.modal');
    modals.forEach(function(modal) {
        modal.addEventListener('show.bs.modal', function() {
            const form = modal.querySelector('form');
            if (form) {
                // Reset form validation
                form.classList.remove('was-validated');
                
                // Clear previous error messages
                const errorMessages = form.querySelectorAll('.invalid-feedback');
                errorMessages.forEach(function(msg) {
                    msg.remove();
                });
            }
        });
    });
    
    // API Key display and copy functionality
    const apiKeyInputs = document.querySelectorAll('input[type="text"][readonly]');
    apiKeyInputs.forEach(function(input) {
        if (input.value && input.value.length > 20) {
            // Add copy button
            const copyBtn = document.createElement('button');
            copyBtn.className = 'btn btn-outline-secondary btn-sm ms-2';
            copyBtn.innerHTML = '<i class="fas fa-copy"></i>';
            copyBtn.type = 'button';
            copyBtn.title = 'Copy API Key';
            
            copyBtn.addEventListener('click', function() {
                input.select();
                document.execCommand('copy');
                
                // Show feedback
                const originalIcon = copyBtn.innerHTML;
                copyBtn.innerHTML = '<i class="fas fa-check text-success"></i>';
                setTimeout(function() {
                    copyBtn.innerHTML = originalIcon;
                }, 1000);
            });
            
            input.parentNode.appendChild(copyBtn);
        }
    });
    
    // Table sorting functionality
    const sortableHeaders = document.querySelectorAll('.table th[data-sort]');
    sortableHeaders.forEach(function(header) {
        header.style.cursor = 'pointer';
        header.innerHTML += ' <i class="fas fa-sort text-muted"></i>';
        
        header.addEventListener('click', function() {
            const table = this.closest('table');
            const tbody = table.querySelector('tbody');
            const rows = Array.from(tbody.querySelectorAll('tr'));
            const columnIndex = Array.from(this.parentNode.children).indexOf(this);
            const sortType = this.getAttribute('data-sort');
            
            // Toggle sort direction
            const isAsc = !this.classList.contains('sort-desc');
            
            // Remove sort classes from all headers
            sortableHeaders.forEach(function(h) {
                h.classList.remove('sort-asc', 'sort-desc');
                h.querySelector('i').className = 'fas fa-sort text-muted';
            });
            
            // Add sort class to current header
            this.classList.add(isAsc ? 'sort-asc' : 'sort-desc');
            this.querySelector('i').className = isAsc ? 'fas fa-sort-up' : 'fas fa-sort-down';
            
            // Sort rows
            rows.sort(function(a, b) {
                const aValue = a.children[columnIndex].textContent.trim();
                const bValue = b.children[columnIndex].textContent.trim();
                
                let comparison = 0;
                if (sortType === 'date') {
                    comparison = new Date(aValue) - new Date(bValue);
                } else if (sortType === 'number') {
                    comparison = parseFloat(aValue) - parseFloat(bValue);
                } else {
                    comparison = aValue.localeCompare(bValue);
                }
                
                return isAsc ? comparison : -comparison;
            });
            
            // Reorder rows in DOM
            rows.forEach(function(row) {
                tbody.appendChild(row);
            });
        });
    });
    
    // Search functionality for tables
    const searchInputs = document.querySelectorAll('input[data-search-table]');
    searchInputs.forEach(function(input) {
        const tableId = input.getAttribute('data-search-table');
        const table = document.getElementById(tableId);
        
        if (table) {
            input.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const rows = table.querySelectorAll('tbody tr');
                
                rows.forEach(function(row) {
                    const text = row.textContent.toLowerCase();
                    row.style.display = text.includes(searchTerm) ? '' : 'none';
                });
            });
        }
    });
    
    // Modal confirmations replace confirm() prompts - handled by Bootstrap modals
    
    // Auto-refresh functionality for dashboard
    if (window.location.pathname === '/') {
        let refreshInterval = setInterval(function() {
            // Only refresh if user is active (not idle)
            if (document.hasFocus()) {
                const statsCards = document.querySelectorAll('.stats-card');
                statsCards.forEach(function(card) {
                    card.style.opacity = '0.7';
                });
                
                // Refresh page content (you could make this AJAX-based)
                setTimeout(function() {
                    location.reload();
                }, 30000); // Refresh every 30 seconds
            }
        }, 30000);
        
        // Stop refresh when user navigates away
        window.addEventListener('beforeunload', function() {
            clearInterval(refreshInterval);
        });
    }
    
    // Tooltip initialization
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.forEach(function(tooltipTriggerEl) {
        new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Popover initialization
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.forEach(function(popoverTriggerEl) {
        new bootstrap.Popover(popoverTriggerEl);
    });
    
    // Form input enhancements
    const formInputs = document.querySelectorAll('.form-control');
    formInputs.forEach(function(input) {
        // Add focus effects
        input.addEventListener('focus', function() {
            this.closest('.mb-3')?.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
            this.closest('.mb-3')?.classList.remove('focused');
        });
        
        // Auto-resize textareas
        if (input.tagName === 'TEXTAREA') {
            input.addEventListener('input', function() {
                this.style.height = 'auto';
                this.style.height = (this.scrollHeight) + 'px';
            });
        }
    });
    
    // Settings form enhancements
    const settingsForm = document.querySelector('#settingsTabs ~ form');
    if (settingsForm) {
        // Add change indicators
        const inputs = settingsForm.querySelectorAll('input, select, textarea');
        inputs.forEach(function(input) {
            const originalValue = input.value;
            
            input.addEventListener('change', function() {
                if (this.value !== originalValue) {
                    this.style.borderColor = '#f39c12';
                    this.style.backgroundColor = '#fff3cd';
                } else {
                    this.style.borderColor = '';
                    this.style.backgroundColor = '';
                }
            });
        });
    }
    
    // Progress bars animation
    const progressBars = document.querySelectorAll('.progress-bar');
    progressBars.forEach(function(bar) {
        const width = bar.getAttribute('data-width') || bar.style.width;
        bar.style.width = '0%';
        
        setTimeout(function() {
            bar.style.width = width;
            bar.style.transition = 'width 1s ease-in-out';
        }, 100);
    });
    
    // Navbar user dropdown enhancement
    const userDropdown = document.querySelector('.navbar-text');
    if (userDropdown) {
        userDropdown.style.cursor = 'pointer';
        userDropdown.addEventListener('click', function() {
            // Could add user menu functionality here
        });
    }
    
    // Mobile responsive enhancements
    function handleResize() {
        if (window.innerWidth <= 768) {
            // Mobile view adjustments
            const tables = document.querySelectorAll('.table-responsive');
            tables.forEach(function(table) {
                table.style.fontSize = '0.875rem';
            });
        } else {
            // Desktop view adjustments
            const tables = document.querySelectorAll('.table-responsive');
            tables.forEach(function(table) {
                table.style.fontSize = '';
            });
        }
    }
    
    window.addEventListener('resize', handleResize);
    handleResize(); // Initial call
    
    // Export functionality enhancement
    const exportButtons = document.querySelectorAll('a[href*="export"]');
    exportButtons.forEach(function(btn) {
        btn.addEventListener('click', function(e) {
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Exporting...';
            btn.style.pointerEvents = 'none';
            
            setTimeout(function() {
                btn.innerHTML = originalText;
                btn.style.pointerEvents = '';
            }, 3000);
        });
    });
    
    // Settings validation
    const settingsInputs = document.querySelectorAll('#settingsTabs input[type="url"], #settingsTabs input[type="email"]');
    settingsInputs.forEach(function(input) {
        input.addEventListener('blur', function() {
            if (this.value && !this.checkValidity()) {
                this.style.borderColor = '#e74c3c';
                
                // Add error message
                let errorMsg = this.parentNode.querySelector('.invalid-feedback');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'invalid-feedback';
                    this.parentNode.appendChild(errorMsg);
                }
                
                if (this.type === 'url') {
                    errorMsg.textContent = 'Please enter a valid URL';
                } else if (this.type === 'email') {
                    errorMsg.textContent = 'Please enter a valid email address';
                }
                
                errorMsg.style.display = 'block';
            } else {
                this.style.borderColor = '';
                const errorMsg = this.parentNode.querySelector('.invalid-feedback');
                if (errorMsg) {
                    errorMsg.style.display = 'none';
                }
            }
        });
    });
    
    // Deployment form validation
    const deploymentForms = document.querySelectorAll('form[action*="deploy"]');
    deploymentForms.forEach(function(form) {
        form.addEventListener('submit', function(e) {
            const maxAttempts = form.querySelector('input[name="max_attempts"]');
            const redirectUrl = form.querySelector('input[name="redirect_url"]');
            
            if (maxAttempts && (maxAttempts.value < 1 || maxAttempts.value > 10)) {
                e.preventDefault();
                alert('Maximum attempts must be between 1 and 10');
                maxAttempts.focus();
                return;
            }
            
            if (redirectUrl && redirectUrl.value && !redirectUrl.checkValidity()) {
                e.preventDefault();
                alert('Please enter a valid redirect URL');
                redirectUrl.focus();
                return;
            }
        });
    });
    
    // Initialize all components
    console.log('Admin panel initialized successfully');
});

// Utility functions
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '9999';
    notification.style.minWidth = '300px';
    
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(function() {
        notification.remove();
    }, 5000);
}

function formatBytes(bytes, decimals = 2) {
    if (bytes === 0) return '0 Bytes';
    
    const k = 1024;
    const dm = decimals < 0 ? 0 : decimals;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
    
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

function timeAgo(date) {
    const now = new Date();
    const diff = now - new Date(date);
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(minutes / 60);
    const days = Math.floor(hours / 24);
    
    if (days > 0) return `${days} day${days > 1 ? 's' : ''} ago`;
    if (hours > 0) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
    if (minutes > 0) return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
    return 'Just now';
}

// Make functions available globally
window.AdminPanel = {
    showNotification: showNotification,
    formatBytes: formatBytes,
    timeAgo: timeAgo
};
